// server/server.js
const express = require('express');
const fs = require('fs');
const path = require('path');
const crypto = require('crypto');
const db = require('./config/database'); // Update ke MySQL config

const app = express();
const PORT = process.env.PORT || 4000;

// Middleware untuk parsing JSON
app.use(express.json());
app.use(express.static(path.join(__dirname, '../public')));

// === AUTHENTICATION SYSTEM ===

// Simple in-memory session store (untuk production, gunakan Redis atau database)
const sessions = new Map();

// Admin credentials (untuk production, simpan di database dengan hash)
const ADMIN_CREDENTIALS = {
    username: 'admin',
    password: 'admin123' // Password sederhana untuk demo
};

// Generate session token
function generateToken() {
    return crypto.randomBytes(32).toString('hex');
}

// Middleware untuk autentikasi admin
function authenticateAdmin(req, res, next) {
    const authHeader = req.headers.authorization;
    
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
        return res.status(401).json({ message: 'Token tidak ditemukan' });
    }
    
    const token = authHeader.substring(7);
    
    if (!sessions.has(token)) {
        return res.status(401).json({ message: 'Token tidak valid' });
    }
    
    const session = sessions.get(token);
    
    // Check if session is expired (24 hours)
    if (Date.now() - session.createdAt > 24 * 60 * 60 * 1000) {
        sessions.delete(token);
        return res.status(401).json({ message: 'Session expired' });
    }
    
    req.admin = session.user;
    next();
}

// Login endpoint
app.post('/api/admin/login', (req, res) => {
    const { username, password } = req.body;
    
    if (!username || !password) {
        return res.status(400).json({ message: 'Username dan password diperlukan' });
    }
    
    if (username === ADMIN_CREDENTIALS.username && password === ADMIN_CREDENTIALS.password) {
        const token = generateToken();
        const session = {
            user: username,
            createdAt: Date.now()
        };
        
        sessions.set(token, session);
        
        res.json({
            message: 'Login berhasil',
            token: token,
            user: username
        });
    } else {
        res.status(401).json({ message: 'Username atau password salah' });
    }
});

// Verify token endpoint
app.get('/api/admin/verify', authenticateAdmin, (req, res) => {
    res.json({
        message: 'Token valid',
        user: req.admin
    });
});

// Logout endpoint
app.post('/api/admin/logout', (req, res) => {
    const authHeader = req.headers.authorization;
    
    if (authHeader && authHeader.startsWith('Bearer ')) {
        const token = authHeader.substring(7);
        sessions.delete(token);
    }
    
    res.json({ message: 'Logout berhasil' });
});

// Endpoint untuk ambil data JSON
app.get('/api/issues', (req, res) => {
    const filePath = path.join(__dirname, '../issues.json');
    fs.readFile(filePath, 'utf8', (err, data) => {
        if (err) return res.status(500).send({ error: "Gagal baca file" });
        res.header("Content-Type", "application/json");
        res.send(data);
    });
});

// Endpoint untuk waktu pembaruan terakhir
app.get('/api/last-updated', (req, res) => {
    fs.stat(path.join(__dirname, '../issues.json'), (err, stats) => {
        if (err) return res.status(500).json({ error: "File tidak ditemukan" });
        res.json({
            lastUpdated: new Date(stats.mtime).toLocaleString('id-ID', { timeZone: 'Asia/Jakarta' })
        });
    });
});

// === PUBLIC API ENDPOINTS ===

// GET semua countries (public endpoint untuk filter)
app.get('/api/countries', async (req, res) => {
    try {
        const rows = await db.query("SELECT id, name, country_code FROM country ORDER BY name");
        res.json(rows);
    } catch (err) {
        console.error("Error fetching countries:", err);
        res.status(500).json({ error: err.message });
    }
});

// === ADMIN API ENDPOINTS ===

// GET semua countries (admin endpoint)
app.get('/api/admin/countries', authenticateAdmin, async (req, res) => {
    try {
        const rows = await db.query("SELECT * FROM country ORDER BY name");
        res.json(rows);
    } catch (err) {
        console.error("Error fetching countries:", err);
        res.status(500).json({ error: err.message });
    }
});

// POST tambah country baru
app.post('/api/admin/countries', authenticateAdmin, async (req, res) => {
    const { name, country_code } = req.body;

    if (!name || !country_code) {
        return res.status(400).json({ error: 'Name and country_code are required' });
    }

    try {
        const result = await db.execute("INSERT INTO country (name, country_code) VALUES (?, ?)", [name, country_code]);
        res.json({ id: result.insertId, name, country_code });
    } catch (err) {
        if (err.code === 'ER_DUP_ENTRY') {
            return res.status(400).json({ error: 'Country code already exists' });
        }
        console.error("Error inserting country:", err);
        res.status(500).json({ error: err.message });
    }
});

// DELETE country
app.delete('/api/admin/countries/:id', authenticateAdmin, async (req, res) => {
    const id = req.params.id;

    try {
        const result = await db.execute("DELETE FROM country WHERE id = ?", [id]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Country not found' });
        }
        res.json({ message: 'Country deleted successfully' });
    } catch (err) {
        console.error("Error deleting country:", err);
        res.status(500).json({ error: err.message });
    }
});

// === UPDATE FUNGSI PENGHITUNGAN NEGARA DI ENDPOINT STATS ===
// Endpoint untuk statistik yang diperlukan oleh stats.html
app.get('/api/stats/countries', async (req, res) => {
    try {
        const filePath = path.join(__dirname, '../issues.json');
        const data = await fs.promises.readFile(filePath, 'utf8');
        
        let issues;
        try {
            issues = JSON.parse(data);
        } catch (parseErr) {
            console.error("Error parsing issues.json:", parseErr);
            return res.status(500).json({ error: "Format file issues tidak valid" });
        }

        // Ambil countries dari database
        const rows = await db.query("SELECT id, name, country_code FROM country");
        
        const countries = {};
        rows.forEach(row => {
            // Gunakan country_code untuk pencarian, tapi tampilkan name
            countries[row.country_code.toLowerCase()] = row.name;
            countries[row.name.toLowerCase()] = row.name;
        });

        // Hitung issues per negara
        const countryCounts = {};
        issues.forEach(issue => {
            const summary = issue.summary?.toLowerCase() || '';
            let found = false;
            for (const [key, name] of Object.entries(countries)) {
                if (summary.includes(key)) {
                    countryCounts[name] = (countryCounts[name] || 0) + 1;
                    found = true;
                    // Jika satu issue bisa terkait lebih dari satu negara, hapus 'break;'
                    // break;
                }
            }
            // Opsional: Hitung issue tanpa negara yang dikenali
            // if (!found) {
            //     countryCounts['Other/Unknown'] = (countryCounts['Other/Unknown'] || 0) + 1;
            // }
        });

        res.json(countryCounts);
    } catch (err) {
        console.error("Error reading issues.json:", err);
        res.status(500).json({ error: "Gagal baca file issues" });
    }
});

// Endpoint untuk statistik country & status yang diperlukan oleh stats.html
app.get('/api/stats/countries-status', async (req, res) => {
    try {
        const filePath = path.join(__dirname, '../issues.json');
        const data = await fs.promises.readFile(filePath, 'utf8');
        
        let issues;
        try {
            issues = JSON.parse(data);
        } catch (parseErr) {
            console.error("Error parsing issues.json:", parseErr);
            return res.status(500).json({ error: "Format file issues tidak valid" });
        }

        // Ambil countries dari database
        const rows = await db.query("SELECT id, name, country_code FROM country");
        
        const countries = {};
        rows.forEach(row => {
            // Gunakan country_code untuk pencarian, tapi tampilkan name
            countries[row.country_code.toLowerCase()] = row.name;
            countries[row.name.toLowerCase()] = row.name;
        });

        // Hitung issues per negara dan status
        const countryStatusCounts = {};
        issues.forEach(issue => {
            const summary = issue.summary?.toLowerCase() || '';
            let foundCountryKey = null;

            for (const [key, name] of Object.entries(countries)) {
                if (summary.includes(key)) {
                    foundCountryKey = key;
                    break; // Hentikan pencarian setelah menemukan satu
                }
            }

            if (foundCountryKey) {
                const countryName = countries[foundCountryKey];
                const status = issue.status?.toLowerCase().trim() || 'unknown';

                if (!countryStatusCounts[countryName]) {
                    countryStatusCounts[countryName] = {};
                }

                countryStatusCounts[countryName][status] = (countryStatusCounts[countryName][status] || 0) + 1;
            }
        });

        res.json(countryStatusCounts);
    } catch (err) {
        console.error("Error reading issues.json:", err);
        res.status(500).json({ error: "Gagal baca file issues" });
    }
});

// Inisialisasi database dan jalankan server
async function startServer() {
    try {
        // Test koneksi database
        const dbConnected = await db.testConnection();
        if (!dbConnected) {
            console.error('❌ Gagal koneksi ke database MySQL');
            process.exit(1);
        }

        // Inisialisasi database (buat tabel jika belum ada)
        const dbInitialized = await db.initializeDatabase();
        if (!dbInitialized) {
            console.error('❌ Gagal inisialisasi database');
            process.exit(1);
        }

        // Jalankan scraping setiap 30 menit
        const scrapeIssues = require('./scrape');
        setInterval(async () => {
            console.log("🔄 Memperbarui data issue...");
            await scrapeIssues();
        }, 1000 * 60 * 30); // 30 menit

        // Jalankan sekali saat server start
        await scrapeIssues();

        app.listen(PORT, () => {
            console.log(`🌐 Server berjalan di http://localhost:${PORT}`);
        });
    } catch (error) {
        console.error('❌ Error starting server:', error);
        process.exit(1);
    }
}

// Jalankan server
startServer();