// server/scrape.js
const puppeteer = require('puppeteer');
const fs = require('fs');
const path = require('path');
const db = require('./config/database'); // Update ke MySQL config

// Fungsi timestamp dalam WIB
function getWIBTimestamp() {
    const now = new Date();
    return new Date(now.getTime() + (7 * 60 * 60 * 1000)).toISOString().replace('T', ' ').split('.')[0];
}

// Log ke file scrape.log
function logToFile(message) {
    const timestamp = getWIBTimestamp();
    const logMessage = `[${timestamp}] ${message}\n`;
    const logPath = path.join(__dirname, '../logs/scrape.log');

    if (!fs.existsSync(path.dirname(logPath))) {
        fs.mkdirSync(path.dirname(logPath), { recursive: true });
    }

    fs.appendFileSync(logPath, logMessage);
}

let isScraping = false; // 🔒 Cegah eksekusi ganda

async function scrapeIssues() {
    if (isScraping) {
        logToFile("⏳ Scraping sedang berjalan, skip...");
        return;
    }

    isScraping = true;

    let browser;
    try {
        logToFile("🔐 Membuka halaman login...");

        // Launch browser dengan args aman
        browser = await puppeteer.launch({
            headless: true,
            args: [
                '--no-sandbox',
                '--disable-setuid-sandbox',
                '--disable-features=NetworkService',
                '--disable-network-conditions'
            ]
        });

        const page = await browser.newPage();
        await page.emulateTimezone('Asia/Jakarta');
        await page.setUserAgent("Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/129.0.0.0 Safari/537.36");

        // Muat cookie jika tersedia
        try {
            const cookies = JSON.parse(fs.readFileSync(path.join(__dirname, '../cookies.json')));
            await page.setCookie(...cookies);
        } catch (e) {
            logToFile("⚠️ Tidak ada cookie tersimpan atau cookie sudah kedaluwarsa");
        }

        // Login ke Jira WMO
        let attempts = 0;
        while (attempts < 3) {
            try {
                await page.goto('https://jira.wmo.int/plugins/servlet/samlsso?idp=2', {
                    waitUntil: 'domcontentloaded',
                    timeout: 90000
                });

                await page.waitForSelector('#signInName', { timeout: 10000 });
                // *** PERINGATAN: Ganti dengan environment variables ***
                await page.type('#signInName', process.env.JIRA_USERNAME || 'iqbal@bmkg.go.id', { delay: 50 });
                await page.type('#password', process.env.JIRA_PASSWORD || 'iqbal@123', { delay: 50 });
                // *** AKHIR PERINGATAN ***

                await Promise.all([
                    page.click('#next'),
                    page.waitForNavigation({ waitUntil: 'networkidle2', timeout: 120000 })
                ]);

                logToFile("✅ Login berhasil");
                break;
            } catch (e) {
                attempts++;
                logToFile(`🔄 Gagal login (percobaan ke-${attempts}), mencoba ulang...`);
                if (attempts >= 3) {
                    logToFile(`❌ Gagal login setelah 3 kali percobaan: ${e.message}`);
                    await browser.close();
                    isScraping = false;
                    return;
                }
            }
        }

        let allIssues = [];
        const baseJql = encodeURIComponent('reporter in ("RWC Indonesia") ORDER BY created DESC');

        for (let pageNum = 1; pageNum <= 5; pageNum++) {
            const startIndex = (pageNum - 1) * 50;
            const url = `https://jira.wmo.int/issues/?jql=${baseJql}&startIndex=${startIndex}`;
            logToFile(`📄 Mengakses halaman ${pageNum}: ${url}`);

            let gotoAttempts = 0;
            let success = false;

            while (gotoAttempts < 3 && !success) {
                try {
                    await page.goto(url, {
                        waitUntil: 'networkidle2',
                        timeout: 120000
                    });

                    success = true;
                } catch (e) {
                    gotoAttempts++;
                    logToFile(`🔄 Ulangi halaman... (${gotoAttempts}/3)`);
                    if (gotoAttempts >= 3) {
                        logToFile(`🚫 Gagal muat halaman ${pageNum}: ${e.message}`);
                        break;
                    }
                }
            }

            if (!success) continue;

            await new Promise(resolve => setTimeout(resolve, 8000)); // tunggu render JS

            const issues = await page.evaluate(() => {
                const rows = Array.from(document.querySelectorAll('tr.issuerow'));
                return rows.map(row => {
                    const keyEl = row.querySelector('.issuekey a');
                    const summaryEl = row.querySelector('.summary a');
                    const statusEl = row.querySelector('.status .jira-issue-status-lozenge');

                    const assigneeEl = row.querySelector('.assignee .user-hover-replaced');
                    const reporterEl = row.querySelector('.reporter .user-hover-replaced');
                    const priorityImgEl = row.querySelector('.priority img');

                    const createdEl = row.querySelector('.created time');
                    const updatedEl = row.querySelector('.updated time');

                    // ✅ Scrap Description dan WIGOS Issue Category
                    const descriptionEl = row.querySelector('.description');
                    const wigosCategoryEl = row.querySelector('.customfield_10008'); // Sesuaikan kelas CSS sesuai Jira

                    return {
                        key: keyEl ? keyEl.getAttribute('data-issue-key') || keyEl.innerText.trim() : null,
                        summary: summaryEl ? summaryEl.innerText.trim() : null,

                        status: statusEl ? statusEl.innerText.trim() : null,

                        assignee: assigneeEl ? assigneeEl.innerText.trim() : null,
                        reporter: reporterEl ? reporterEl.innerText.trim() : null,
                        priority: priorityImgEl ? priorityImgEl.alt.trim() : null,
                        created: createdEl ? createdEl.innerText.trim() : null,
                        updated: updatedEl ? updatedEl.innerText.trim() : null,

                        // ✅ Field baru
                        description: descriptionEl ? descriptionEl.textContent.trim() : null,
                        wigosIssueCategory: wigosCategoryEl ? wigosCategoryEl.textContent.trim() : null
                    };
                });
            });

            if (issues.length === 0) {
                logToFile(`⚠️ Tidak ada issue ditemukan di halaman ${pageNum}`);
                continue;
            }

            if (issues.length < 50) {
                logToFile(`🚫 Hanya ditemukan ${issues.length} issue di halaman ${pageNum}. Tidak ada halaman berikutnya.`);
                allIssues.push(...issues);
                break;
            }

            const uniqueKeys = new Set(allIssues.map(i => i.key));
            const newIssues = issues.filter(issue => !uniqueKeys.has(issue.key));

            if (newIssues.length > 0) {
                logToFile(`🔍 Ditemukan ${newIssues.length} issue baru di halaman ${pageNum}`);
                allIssues.push(...newIssues);
            } else {
                logToFile(`🔁 Halaman ${pageNum} tidak memiliki issue baru. Berhenti scraping.`);
                break;
            }
        }

        const filePath = path.join(__dirname, '../issues.json');
        fs.writeFileSync(filePath, JSON.stringify(allIssues, null, 2), 'utf-8');
        logToFile(`📦 Total ${allIssues.length} issue disimpan`);

    } catch (err) {
        logToFile(`❌ Error terjadi: ${err.message}`);
    } finally {
        if (browser) await browser.close();
        isScraping = false;
        logToFile("🔚 Scraping selesai.");
    }
}

// Jalankan scraping otomatis setiap jam
setInterval(async () => {
    console.log("🔄 Memperbarui data issue...");
    await scrapeIssues();
}, 30 * 60 * 1000); // waktu update

// Jalankan sekali saat server start
scrapeIssues();

module.exports = scrapeIssues;