// server/server.js
const express = require('express');
const fs = require('fs');
const path = require('path');
const db = require('./database'); // Tambahkan ini

const app = express();
const PORT = process.env.PORT || 5000;

// Middleware untuk parsing JSON
app.use(express.json());
app.use(express.static(path.join(__dirname, '../public')));

// Endpoint untuk ambil data JSON
app.get('/api/issues', (req, res) => {
    const filePath = path.join(__dirname, '../issues.json');
    fs.readFile(filePath, 'utf8', (err, data) => {
        if (err) return res.status(500).send({ error: "Gagal baca file" });
        res.header("Content-Type", "application/json");
        res.send(data);
    });
});

// Endpoint untuk waktu pembaruan terakhir
app.get('/api/last-updated', (req, res) => {
    fs.stat(path.join(__dirname, '../issues.json'), (err, stats) => {
        if (err) return res.status(500).json({ error: "File tidak ditemukan" });
        res.json({
            lastUpdated: new Date(stats.mtime).toLocaleString('id-ID', { timeZone: 'Asia/Jakarta' })
        });
    });
});

// === ADMIN API ENDPOINTS ===

// GET semua countries
app.get('/api/admin/countries', (req, res) => {
    db.all("SELECT * FROM countries ORDER BY name", (err, rows) => {
        if (err) {
            console.error("Error fetching countries:", err);
            return res.status(500).json({ error: err.message });
        }
        res.json(rows);
    });
});

// POST tambah country baru
app.post('/api/admin/countries', (req, res) => {
    const { code, name } = req.body;

    if (!code || !name) {
        return res.status(400).json({ error: 'Code and name are required' });
    }

    db.run("INSERT INTO countries (code, name) VALUES (?, ?)", [code, name], function(err) {
        if (err) {
            if (err.message.includes('UNIQUE constraint failed')) {
                return res.status(400).json({ error: 'Country code already exists' });
            }
            console.error("Error inserting country:", err);
            return res.status(500).json({ error: err.message });
        }
        res.json({ id: this.lastID, code, name });
    });
});

// DELETE country
app.delete('/api/admin/countries/:id', (req, res) => {
    const id = req.params.id;

    db.run("DELETE FROM countries WHERE id = ?", [id], function(err) {
        if (err) {
            console.error("Error deleting country:", err);
            return res.status(500).json({ error: err.message });
        }
        if (this.changes === 0) {
            return res.status(404).json({ error: 'Country not found' });
        }
        res.json({ message: 'Country deleted successfully' });
    });
});

// === UPDATE FUNGSI PENGHITUNGAN NEGARA DI ENDPOINT STATS ===
// Endpoint untuk statistik yang diperlukan oleh stats.html
app.get('/api/stats/countries', (req, res) => {
    const filePath = path.join(__dirname, '../issues.json');
    fs.readFile(filePath, 'utf8', (err, data) => {
        if (err) {
            console.error("Error reading issues.json:", err);
            return res.status(500).json({ error: "Gagal baca file issues" });
        }

        let issues;
        try {
            issues = JSON.parse(data);
        } catch (parseErr) {
            console.error("Error parsing issues.json:", parseErr);
            return res.status(500).json({ error: "Format file issues tidak valid" });
        }

        // Ambil countries dari database
        db.all("SELECT code, name FROM countries", (dbErr, rows) => {
            if (dbErr) {
                console.error("Error fetching countries from DB:", dbErr);
                return res.status(500).json({ error: "Gagal mengambil data negara" });
            }

            const countries = {};
            rows.forEach(row => {
                countries[row.code.toLowerCase()] = row.name;
            });

            // Hitung issues per negara
            const countryCounts = {};
            issues.forEach(issue => {
                const summary = issue.summary?.toLowerCase() || '';
                let found = false;
                for (const [key, name] of Object.entries(countries)) {
                    if (summary.includes(key)) {
                        countryCounts[name] = (countryCounts[name] || 0) + 1;
                        found = true;
                        // Jika satu issue bisa terkait lebih dari satu negara, hapus 'break;'
                        // break;
                    }
                }
                // Opsional: Hitung issue tanpa negara yang dikenali
                // if (!found) {
                //     countryCounts['Other/Unknown'] = (countryCounts['Other/Unknown'] || 0) + 1;
                // }
            });

            res.json(countryCounts);
        });
    });
});

// Endpoint untuk statistik country & status yang diperlukan oleh stats.html
app.get('/api/stats/countries-status', (req, res) => {
    const filePath = path.join(__dirname, '../issues.json');
    fs.readFile(filePath, 'utf8', (err, data) => {
        if (err) {
            console.error("Error reading issues.json:", err);
            return res.status(500).json({ error: "Gagal baca file issues" });
        }

        let issues;
        try {
            issues = JSON.parse(data);
        } catch (parseErr) {
            console.error("Error parsing issues.json:", parseErr);
            return res.status(500).json({ error: "Format file issues tidak valid" });
        }

        // Ambil countries dari database
        db.all("SELECT code, name FROM countries", (dbErr, rows) => {
            if (dbErr) {
                console.error("Error fetching countries from DB:", dbErr);
                return res.status(500).json({ error: "Gagal mengambil data negara" });
            }

            const countries = {};
            rows.forEach(row => {
                countries[row.code.toLowerCase()] = row.name;
            });

            // Hitung issues per negara dan status
            const countryStatusCounts = {};
            issues.forEach(issue => {
                const summary = issue.summary?.toLowerCase() || '';
                let foundCountryKey = null;

                for (const [key, name] of Object.entries(countries)) {
                    if (summary.includes(key)) {
                        foundCountryKey = key;
                        break; // Hentikan pencarian setelah menemukan satu
                    }
                }

                if (foundCountryKey) {
                    const countryName = countries[foundCountryKey];
                    const status = issue.status?.toLowerCase().trim() || 'unknown';

                    if (!countryStatusCounts[countryName]) {
                        countryStatusCounts[countryName] = {};
                    }

                    countryStatusCounts[countryName][status] = (countryStatusCounts[countryName][status] || 0) + 1;
                }
            });

            res.json(countryStatusCounts);
        });
    });
});

// Jalankan scraping setiap 30 menit
const scrapeIssues = require('./scrape');
setInterval(async () => {
    console.log("🔄 Memperbarui data issue...");
    await scrapeIssues();
}, 1000 * 60 * 30); // 30 menit

// Jalankan sekali saat server start
scrapeIssues();

app.listen(PORT, () => {
    console.log(`🌐 Server berjalan di http://localhost:${PORT}`);
});